#!/bin/bash

# Disable immediate exit so we can handle errors properly
set +e

# Color codes for formatting
RED='\033[91m'
GREEN='\033[92m'
MAGENTA='\033[35m'
CYAN='\033[36m'
RESET='\033[0m'

# OS Check
os_name=$(lsb_release -si 2>/dev/null)
os_version=$(lsb_release -sr 2>/dev/null)

if [[ "$os_name" != "Kali" ]]; then
    echo -e "${RED}✖ OS not supported... Please use Kali Linux.${RESET}"
    echo -e "${CYAN}Detected OS: ${os_name} ${os_version}${RESET}"
    exit 1
fi

echo -e "${GREEN}✔ Detected supported OS: ${os_name} ${os_version}${RESET}"

# Get current working directory
cwd=$(pwd)

# Format the string to only include the path to the repo's main file
index=$(echo "$cwd" | awk -F'/MiniNet-framework' '{print length($1)+length("/MiniNet-framework")}')
truncated_cwd=$(echo "$cwd" | cut -c 1-"$index")

# Function for animated spinner
echo_spinner() {
    local pid=$1
    local message=$2
    local symbols=("+" "x")
    echo -ne "[${GREEN}+${RESET}] $message"
    
    while kill -0 $pid 2>/dev/null; do
        for symbol in "${symbols[@]}"; do
            echo -ne "\r[${GREEN}${symbol}${RESET}] $message"
            sleep 0.5
        done
    done

    wait $pid
    exit_code=$?
    if [ $exit_code -ne 0 ]; then
        echo -e "\r[${RED}✖${RESET}] $message (Failed)"
    else
        echo -e "\r[${GREEN}✔${RESET}] $message    "
    fi
}

# Run a command with error detection
run_command() {
    ($1 > /dev/null 2>&1) &
    local pid=$!
    echo_spinner $pid "$2"
}

run_command "sudo apt update -y" "Updating package list..."
run_command "sudo apt install -y ifupdown pip curl aircrack-ng john dsniff tmux git help2man" "Installing required tools..."

# Adding Submodules to safe.directory
run_command "git config --global --add safe.directory \"$truncated_cwd\"" "Adding Submodules to safe.directory..."

# Initialize Submodules
run_command "git submodule init" "Initializing Submodules..."

# Update Submodules
run_command "git submodule update" "Updating Submodules..."

run_command "sudo -E pip config set global.break-system-packages true" "Fixing Package Configurations..."

# Set global pip variable to break system packages
run_command "sudo -E pip install -r requirements.txt" "Configuring pip..."

# Install Mininet
run_command "sudo apt install -y mininet --allow-downgrades" "Installing Mininet..."
run_command "sudo apt install bettercap -y" "Installing Bettercap..."

# Run Install Script
run_command "../mininet-wifi/util/install.sh -n" "Installing Mininet Wifi and core dependencies..."
run_command "../mininet-wifi/util/install.sh -l" "Installing wmediumd..."
run_command "../mininet-wifi/util/install.sh -v" "Installing Openvswitch..."
run_command "../mininet-wifi/util/install.sh -f" "Installing Openflow..."
run_command "../mininet-wifi/util/install.sh -W" "Installing additional dependencies..."
run_command "sudo python3 ../mininet-wifi/setup.py install" "Running Mininet-Wifi Setup..."

run_command "sudo make install" "Compiling"

# Compile
run_command "sudo apt install openvswitch-testcontroller -y" "Installing Openvswitch..."
run_command "sudo ln /usr/bin/ovs-testcontroller /usr/bin/controller" "Configuring Controller..."
run_command "sudo service openvswitch-switch start" "Starting Openvswitch..."
run_command "sudo cp ./main_menu /usr/bin" "Copying Files to Bin..."
run_command "sudo chmod +x /usr/bin/main_menu" "Adjusting main_menu Privelleges..."

echo -e "${MAGENTA}WifiForge Finished Installing! 🎉${RESET}"
